
/*******************************************************************************/
/*  
	This script produces Figure 5
	Mongey, Pilossoph, and Weinberg: "Which Workers Bear the Burden of Social Distancing?"
	Journal of Economic Inequality 2021	
    --
	Alexander Weinberg
	February 16, 2021
*/
/*******************************************************************************/


// ________________________________________
// Load CPS
graph drop _all
use "../Data/CPS/monthly_cps_2010_2020.dta", clear

// ________________________________________
// Monthly employment
collapse (sum) unemployed employed nilf [pw=compwt], by(month year occ_2digit) fast

// ________________________________________
// LWFH* and HPP*
merge m:1 occ_2digit using "../Data/2_digit_pp_wfh_onet" // pp/wfh scores
keep if _merge == 3
drop _merge 

// ________________________________________
// Essential occupations
// Link: https://www.brookings.edu/wp-content/uploads/2020/03/Front-Line-Workers-Appendix.docx
merge m:1 occ_2digit using "../Data/DHS/occ2digit_essential", nogen // share essential
gen cuttoff   	= 0.75
gen essential 	= share_essential >= cuttoff

// ________________________________________
// Time series
gen date 	 	= ym(year,month)
format date %tm
tsset occ_2digit date, monthly

// ________________________________________
// LOG EMP variables
gen E_feb = employed if month==2
gen E_apr = employed if month==4
gen E_aug = employed if month==8

// ________________________________________
// Prepare for log differences
sort occ_2digit year month
by occ_2digit year: replace E_feb = E_feb[_n-1] if mi(E_feb)
by occ_2digit year: replace E_apr = E_apr[_n-1] if mi(E_apr)
keep if month == 8
drop month

// ________________________________________
// Dlog Employment
sort occ_2digit year
bys year: gen dlogE_apr 		= log(E_apr) - log(E_feb) 
bys year: gen dlogE_aug 		= log(E_aug) - log(E_feb) 

bys occ_2digit year: replace dlogE_apr = dlogE_apr[_n-1] if mi(dlogE_apr) 

// ________________________________________
// Avg. dlogE 2010-2019
bys occ_2digit: egen mean_dlogE_apr_10_19 	 = mean(dlogE_apr) 	if (year < 2020)
bys occ_2digit: egen mean_dlogE_aug_10_19 	 = mean(dlogE_aug) 	if (year < 2020)

bys	occ_2digit: replace mean_dlogE_apr_10_19 = mean_dlogE_apr_10_19[_n-1] if mi(mean_dlogE_apr_10_19) 
bys	occ_2digit: replace mean_dlogE_aug_10_19 = mean_dlogE_aug_10_19[_n-1] if mi(mean_dlogE_aug_10_19) 
keep if year==2020

// ________________________________________
// Adj 2020 numbers by decade average
gen dlogE_apr_adj     							 = dlogE_apr - mean_dlogE_apr_10_19
gen dlogE_aug_adj     							 = dlogE_aug - mean_dlogE_aug_10_19

format %2.1f low_wfh pp
format %8.2f dlogE_*_adj

// ------------------------------------------------------
// FEB-APR DLOG E
// ------------------------------------------------------


graph drop _all

// LWFH
twoway ///
	(scatter dlogE_apr_adj low_wfh [aw=tot_emp] if !essential , mc(none) sort mlcolor(blue) mlwidth(medthick)) ///
	(scatter dlogE_apr_adj low_wfh [aw=tot_emp] if essential , mc(none)  sort mlcolor(red) mlwidth(medthick)) ///
	(lfitci dlogE_apr_adj low_wfh if !essential [aw=employed], level(90) ciplot(rline) alcolor(blue) alwidth(medthick) alpattern(_) clcolor(blue) clwidth(vthick)) ///
	, ytitle("Relative change in Feb-Apr log employment", size(medlarge)) ///
	xtitle("Low work-from-home >>" , size(large)) ///
	yscale(range(-0.67 0.20)) ylabel(-0.6(0.2)0.0) ///
	legend(pos(7) cols(1) order(1 "Non-essential" 2 "Essential")  size(12pt)) ///
	xsize(5) ysize(4)  ///
    ///xscale(range(0 1.02)) xlabel(-0.0(0.1)1.0, labsize(medium)) ///
    ///yscale(range(-0.12 0.06)) ylabel(-0.1(0.05)0.05, labsize(medium)) ///
	title("A. Work-from-home (Feb-Apr)") name(Y1)

// PP
twoway ///
	(scatter dlogE_apr_adj pp [aw=tot_emp] if !essential , mc(none) sort mlcolor(blue) mlwidth(medthick)) ///
	(scatter dlogE_apr_adj pp [aw=tot_emp] if essential , mc(none) sort mlcolor(red) mlwidth(medthick)) ///
	(lfitci dlogE_apr_adj pp if !essential [aw=employed], level(90) ciplot(rline) alcolor(blue) alwidth(medthick) alpattern(_) clcolor(blue) clwidth(vthick)) ///
	, ytitle("Relative change in Feb-Apr log employment", size(medlarge)) ///
	xtitle("High physical-proximity >>" , size(large)) ///
	legend(off) ///
	yscale(range(-0.67 0.20)) ylabel(-0.6(0.2)0.0) ///
	xsize(5) ysize(4)  ///
	title("B. Physical-proximity (Feb-Apr)") name(Y2)


graph combine Y1 Y2, xcommon // ycommon 
graph export "../Figures/fig5_apr.eps", replace

// ------------------------------------------------------
// MAY-AUG DLOG E
// ------------------------------------------------------


// LWFH
twoway ///
	(scatter dlogE_aug_adj low_wfh [aw=tot_emp] if !essential , mc(none) sort mlcolor(blue) mlwidth(medthick)) ///
	(scatter dlogE_aug_adj low_wfh [aw=tot_emp] if essential  , mc(none) sort mlcolor(red) mlwidth(medthick)) ///
	(lfitci dlogE_aug_adj low_wfh if !essential [aw=employed], level(90) ciplot(rline) alcolor(blue) alwidth(medthick) alpattern(_) clcolor(blue) clwidth(vthick)) ///
	, ytitle("Relative change in Feb-Aug log employment", size(medlarge)) ///
	xtitle("Low work-from-home >>" , size(large)) ///
	yscale(range(-0.67 0.20)) ylabel(-0.6(0.2)0.00) ///
	legend(pos(7) cols(1) order(1 "Non-essential" 2 "Essential") size(12pt)) ///
	xsize(5) ysize(4)  ///
    ///xscale(range(0 1.02)) xlabel(-0.0(0.1)1.0, labsize(medium)) ///
    ///yscale(range(-0.12 0.06)) ylabel(-0.1(0.05)0.05, labsize(medium)) ///
	title("C. Work-from-home (Feb-Aug)") name(Z1)

// PP
twoway ///
	(scatter dlogE_aug_adj pp [aw=tot_emp] if !essential, mc(none) sort mlcolor(blue) mlwidth(medthick)) ///
	(scatter dlogE_aug_adj pp [aw=tot_emp] if essential , mc(none) sort mlcolor(red) mlwidth(medthick)) ///
	(lfitci dlogE_aug_adj pp if !essential [aw=employed], level(90) ciplot(rline) alcolor(blue) alwidth(medthick) alpattern(_) clcolor(blue) clwidth(vthick)) ///
	, ytitle("Relative change in Feb-Aug log employment", size(medlarge)) ///
	xtitle("High physical-proximity >>" , size(large)) ///
	yscale(range(-0.67 0.20)) ylabel(-0.6(0.2)0.00) ///
	legend(off) ///
	xsize(5) ysize(4)  ///
    ///xscale(range(0 1.02)) xlabel(-0.0(0.1)1.0, labsize(medium)) ///
    ///yscale(range(-0.12 0.06)) ylabel(-0.1(0.05)0.05, labsize(medium)) ///
	title("D. Physical-proximity (Feb-Aug)") name(Z2)

graph combine Z1 Z2, xcommon
graph export "../Figures/fig5_aug.eps", replace




//end
